/**
 * ProverOchka Publisher - Pixso Plugin
 * Sandbox code that handles node serialization and image export
 */

// Show UI
pixso.showUI(__html__, { width: 320, height: 400 });

/**
 * Serialize a Pixso node to JSON structure compatible with backend parser
 * @param {SceneNode} node - The node to serialize
 * @param {number} rootFrameX - X coordinate of the root frame (for normalization)
 * @param {number} rootFrameY - Y coordinate of the root frame (for normalization)
 */
function serializeNode(node, rootFrameX = 0, rootFrameY = 0) {
  const serialized = {
    id: node.id,
    name: node.name,
    type: node.type,
    visible: node.visible !== false
  };

  // Extract absolute bounding box and normalize coordinates relative to root frame
  if (node.absoluteBoundingBox) {
    // Calculate relative coordinates (normalized to 0,0 based on root frame)
    const absX = node.absoluteBoundingBox.x - rootFrameX;
    const absY = node.absoluteBoundingBox.y - rootFrameY;
    
    serialized.absoluteBoundingBox = {
      x: absX, // Use calculated relative X (normalized to root frame)
      y: absY, // Use calculated relative Y (normalized to root frame)
      width: node.absoluteBoundingBox.width,
      height: node.absoluteBoundingBox.height
    };
  }

  // Extract absolute render bounds and normalize coordinates relative to root frame
  if (node.absoluteRenderBounds) {
    // Calculate relative coordinates (normalized to 0,0 based on root frame)
    const renderX = node.absoluteRenderBounds.x - rootFrameX;
    const renderY = node.absoluteRenderBounds.y - rootFrameY;
    
    serialized.absoluteRenderBounds = {
      x: renderX, // Use calculated relative X (normalized to root frame)
      y: renderY, // Use calculated relative Y (normalized to root frame)
      width: node.absoluteRenderBounds.width,
      height: node.absoluteRenderBounds.height
    };
  }

  // Extract fills (simplified)
  if (node.fills && Array.isArray(node.fills)) {
    serialized.fills = node.fills
      .filter(fill => fill.visible !== false)
      .map(fill => {
        if (fill.type === 'SOLID') {
          return {
            type: 'SOLID',
            color: {
              r: fill.color.r,
              g: fill.color.g,
              b: fill.color.b
            },
            opacity: fill.opacity !== undefined ? fill.opacity : 1.0
          };
        }
        // For other fill types, return basic structure
        return {
          type: fill.type,
          visible: fill.visible !== false
        };
      });
  }

  // Extract strokes
  if (node.strokes && Array.isArray(node.strokes)) {
    serialized.strokes = node.strokes
      .filter(stroke => stroke.visible !== false)
      .map(stroke => {
        if (stroke.type === 'SOLID') {
          return {
            type: 'SOLID',
            color: {
              r: stroke.color.r,
              g: stroke.color.g,
              b: stroke.color.b
            },
            opacity: stroke.opacity !== undefined ? stroke.opacity : 1.0
          };
        }
        return {
          type: stroke.type,
          visible: stroke.visible !== false
        };
      });
  }

  // Extract effects (shadows, blurs, etc.)
  if (node.effects && Array.isArray(node.effects)) {
    serialized.effects = node.effects
      .filter(effect => effect.visible !== false)
      .map(effect => ({
        type: effect.type,
        visible: effect.visible !== false
      }));
  }

  // Extract text content (for TEXT nodes)
  if (node.type === 'TEXT' && node.characters) {
    serialized.characters = node.characters;
    
    // Extract font size from style
    if (node.fontSize) {
      serialized.fontSize = node.fontSize;
    } else if (node.fontName) {
      // Try to get fontSize from text style
      serialized.style = {
        fontSize: node.fontSize || 16
      };
    }
  }

  // Recursively serialize children
  // Pass rootFrameX and rootFrameY so all children coordinates are normalized relative to root frame
  if (node.children && Array.isArray(node.children)) {
    serialized.children = node.children
      .filter(child => child.visible !== false)
      .map(child => serializeNode(child, rootFrameX, rootFrameY));
  }

  return serialized;
}

/**
 * Wrap serialized frames in Pixso API-compatible structure
 * @param {string} pageName - Name of the current page
 * @param {Array} framesData - Array of serialized frame data objects
 * @param {Array} framesMeta - Array of frame metadata objects {id, name, absoluteBoundingBox}
 */
function wrapInDocumentStructure(pageName, framesData, framesMeta) {
  return {
    document: {
      id: "0:0",
      name: "Document",
      type: "DOCUMENT",
      children: [
        {
          id: "0:1",
          name: pageName,
          type: "CANVAS",
          children: framesData
        }
      ]
    },
    frames: framesMeta.map(frameMeta => ({
      id: frameMeta.id,
      name: frameMeta.name,
      image: frameMeta.id.replace(/:/g, "_") + ".png",
      // Root frame coordinates are normalized to (0, 0) relative to itself
      absoluteBoundingBox: frameMeta.absoluteBoundingBox ? {
        x: 0, // Root frame is at (0, 0) relative to itself
        y: 0, // Root frame is at (0, 0) relative to itself
        width: frameMeta.absoluteBoundingBox.width,
        height: frameMeta.absoluteBoundingBox.height
      } : null
    }))
  };
}

// Listen for export request from UI
pixso.ui.onmessage = async (msg) => {
  if (msg.type === 'export-request') {
    try {
      // Get current page
      const currentPage = pixso.currentPage;
      const pageName = currentPage.name;

      // Filter top-level children: only visible FRAME, SECTION, or COMPONENT nodes
      const validNodes = currentPage.children.filter(node => {
        const isValidType = node.type === 'FRAME' || node.type === 'SECTION' || node.type === 'COMPONENT';
        const isVisible = node.visible !== false;
        return isValidType && isVisible;
      });

      if (validNodes.length === 0) {
        pixso.ui.postMessage({
          type: 'export-error',
          error: 'No visible Frames, Sections, or Components found on this page. Please add at least one.'
        });
        return;
      }

      pixso.ui.postMessage({
        type: 'export-progress',
        message: `Found ${validNodes.length} frame(s). Exporting...`
      });

      // Batch export: export all valid nodes
      const assets = [];
      const framesData = [];
      const framesMeta = [];

      for (let i = 0; i < validNodes.length; i++) {
        const node = validNodes[i];
        
        pixso.ui.postMessage({
          type: 'export-progress',
          message: `Exporting ${i + 1}/${validNodes.length}: ${node.name}...`
        });

        // Export image
        const imageBytes = await node.exportAsync({
          format: 'PNG',
          constraint: {
            type: 'SCALE',
            value: 1
          }
        });

        // Get root frame coordinates for normalization
        const rootFrameX = node.absoluteBoundingBox ? node.absoluteBoundingBox.x : 0;
        const rootFrameY = node.absoluteBoundingBox ? node.absoluteBoundingBox.y : 0;
        
        // Serialize with normalization: root frame becomes (0,0), all children are relative to it
        const serializedFrame = serializeNode(node, rootFrameX, rootFrameY);

        // Store asset (image bytes)
        assets.push({
          id: node.id,
          bytes: imageBytes
        });

        // Store frame data for JSON structure
        framesData.push(serializedFrame);

        // Store frame metadata
        framesMeta.push({
          id: node.id,
          name: node.name,
          absoluteBoundingBox: node.absoluteBoundingBox
        });
      }

      // Create JSON structure: Document -> Canvas (Current Page) -> [List of Frames]
      const structure = wrapInDocumentStructure(pageName, framesData, framesMeta);

      // Send success message with batch data
      pixso.ui.postMessage({
        type: 'export-success',
        name: pageName,
        structure: structure,
        assets: assets
      });

    } catch (error) {
      pixso.ui.postMessage({
        type: 'export-error',
        error: `Export failed: ${error.message}`
      });
    }
  }
};

